#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
fi

DEFAULT="${PERSISTENT_DIRECTORY}/../image/factory_defaults/etc/network/interfaces"
CURRENT="${PERSISTENT_DIRECTORY}/etc/network/interfaces"
# Check new platform incompatible
COUNT_BR0_DEFAULT=$(grep "iface br0" ${DEFAULT} | wc -l )
COUNT_BR0_CURRENT=$(grep "iface br0" ${CURRENT} | wc -l )
COUNT_ETH0_DEFAULT=$(grep "iface eth0" ${DEFAULT} | wc -l )
COUNT_ETH0_CURRENT=$(grep "iface eth0" ${CURRENT} | wc -l )
COUNT_ETH0_2_DEFAULT=$(grep "iface eth0.2" ${DEFAULT} | wc -l )
# Check yocto4 incompatible
BR0_PREUP_DEFAULT=$(cat ${DEFAULT} | grep 'iface br0' -A1 | grep 'pre-up' | awk -F ' ' '{print $4}')
BR0_PREUP_CURRENT=$(cat ${CURRENT} | grep 'iface br0' -A1 | grep 'pre-up' | awk -F ' ' '{print $4}')
# Check R11 incompatible
GENERATE_BR0_JSON=false

if [[ -f "/proc/device-tree/is_module" ]] ||
   [[ ! -f "/proc/device-tree/is_chassis" ]]; then
  if [[ ${RUNNING_RELEASE_VERSION} =~ ^([0-9]+).[0-9]+.[0-9]+ ]]; then
    MAJOR=${BASH_REMATCH[1]}
    # Preserve network config file which is written by ILA (since R8)
    if (( MAJOR >= 8 )); then
      LOG "Preserve network interfaces file for slot module!"
    elif [[ -s ${CURRENT} ]]; then
      LOG "Empty network interfaces file for slot module!"
      echo -n "" > ${CURRENT}
    fi
  else
    LOG "Invalid version: ${RUNNING_RELEASE_VERSION}"
  fi
elif [[ ${COUNT_BR0_DEFAULT}  -ne ${COUNT_BR0_CURRENT}   ]] ||
     [[ ${COUNT_ETH0_DEFAULT} -ne ${COUNT_ETH0_CURRENT}  ]] ||
     [[ "${BR0_PREUP_DEFAULT}" != "${BR0_PREUP_CURRENT}" ]]; then
  LOG "Incompatible network interface detected!"
  cp -f ${DEFAULT} ${CURRENT}

  if [[ -f /proc/device-tree/dcpr ]] || [[ -f /proc/device-tree/dcpm2 ]]; then
    ifconfig=$(ifconfig eth0.2 | grep "inet addr" | xargs)
    ipaddr=$(echo ${ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    netmask=$(echo ${ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    gateway=$(ip route show eth0.2 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
    prefixlen=$(/sbin/ip -o -4 address show eth0.2 | awk '{print $4}' | cut -d '/' -f2)
    LOG "Restore defaults network interface, keep eth0.2 network settings: [${ipaddr}]/[${prefixlen}] [${netmask}] [${gateway}]"

    # New method for network configuration since R11
    dcpr_nw_if='eth0.2'
    if [[ ${COUNT_ETH0_2_DEFAULT} == 0 ]]; then
      GENERATE_BR0_JSON=true
      LOG "No eth0.2 in default network template file, use br0 instead"
      dcpr_nw_if='br0'
    fi

    # Fresh DCP chassis from factory do not have gateway configured
    if [[ -z ${gateway} ]]; then
      sed -i '/./{H;$!d} ; x ; s/\(iface '${dcpr_nw_if}' inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'/' ${CURRENT}
    else
      sed -i '/./{H;$!d} ; x ; s/\(iface '${dcpr_nw_if}' inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'\n\tgateway '${gateway}'/' ${CURRENT}
    fi
    # delete all leading blank lines at top of file
    sed -i '/./,$!d' ${CURRENT}
  elif [[ -f "/proc/device-tree/is_chassis" ]]; then
    br0_ifconfig=$(ifconfig br0 | grep "inet addr" | xargs)
    ipaddr=$(echo ${br0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    netmask=$(echo ${br0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    gateway=$(ip route show br0 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
    prefixlen=$(/sbin/ip -o -4 address show br0 | awk '{print $4}' | cut -d '/' -f2)
    LOG "Restore defaults network interface, keep br0 network settings: [${ipaddr}]/[${prefixlen}] [${netmask}] [${gateway}]"

    # New method for network configuration since R11
    if [[ ${COUNT_ETH0_2_DEFAULT} == 0 ]]; then
      GENERATE_BR0_JSON=true
    fi

    if [[ -z "${BR0_PREUP_DEFAULT}" ]]; then
      # Yocto4
      if [[ -z ${gateway} ]]; then
        sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'/' ${CURRENT}
      else
        sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'\n\tgateway '${gateway}'/' ${CURRENT}
      fi
    else
      # Fresh DCP chassis from factory do not have gateway configured
      if [[ -z ${gateway} ]]; then
        sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n.*\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'/' ${CURRENT}
      else
        sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n.*\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'\n\tgateway '${gateway}'/' ${CURRENT}
      fi
    fi

    # eth0 does not have any gateway configured
    eth0_ifconfig=$(ifconfig eth0 | grep "inet addr" | xargs)
    eth0_ipaddr=$(echo ${eth0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    eth0_netmask=$(echo ${eth0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    LOG "Restore defaults network interface, keep eth0 network settings: [${eth0_ipaddr}] [${eth0_netmask}]"
    sed -i '/./{H;$!d} ; x ; s/\(iface eth0 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${eth0_ipaddr}'\n\tnetmask '${eth0_netmask}'/' ${CURRENT}

    # delete all leading blank lines at top of file
    sed -i '/./,$!d' ${CURRENT}
  else
    LOG "Restore defaults network interface."
  fi
fi

# New method for network configuration since R11
if [[ $GENERATE_BR0_JSON == true ]]; then
  br0_json_string=$(jq -c -n \
                    '{"ifname":"br0","operstate":"UP","addr_info":[{"local":$ipaddr,"prefixlen":$prefixlen}],"route":[{"dst":"default","gateway":$gateway,"flags":[]}]}' \
                    --arg     ipaddr    "$ipaddr" \
                    --argjson prefixlen "$prefixlen" \
                    --arg     gateway   "$gateway")
  LOG "Generating br0.json: ${br0_json_string}"
  br0_json_file_path="${PERSISTENT_DIRECTORY}/etc/network/br0.json"
  echo "${br0_json_string}" > ${br0_json_file_path}
fi
