#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to firmware directory
  -h, --help                       display help
EOM

	exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

FIRMWARE_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  FIRMWARE_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$FIRMWARE_DIRECTORY" ]
then
  LOG "The directory $FIRMWARE_DIRECTORY does not exist."
  exit 1
fi

#Upgrade firmware
INSTALLING_FIRMWARE_VERSION=0.4.39
INSTALLING_FIRMWARE_FILE="${FIRMWARE_DIRECTORY}/vsc7512-${INSTALLING_FIRMWARE_VERSION}.tar"
VSC_DH_FILE="/tmp/device_handlers/vsc751x"
RUNNING_FIRMWARE_VERSION=""
RECONFIG_TUNNEL_AFTER_FWUPGRADE_RETRY=10

fail=0

if [ -S "$VSC_DH_FILE" ]
then

  FW_VER_PATH="data/fwVersion"
  [[ -f /usr/sbin/smartmanager/smartmanager.js ]] && FW_VER_PATH="data/runningFirmwareVersion"
  RUNNING_FIRMWARE_VERSION=$(tiny_dh_client -d "$VSC_DH_FILE" -c gets -a ${FW_VER_PATH})

  # Don't upgrade DCP-F A22, R22, DE22 from 0.4.30 (Bypass this check for VG which has sfp_tx_force_enable-gpios pin)
  if [[ -f /proc/device-tree/dcpf && ! -f /proc/device-tree/sfp0/sfp_tx_force_enable-gpios ]]
  then
    RUNNING_FIRMWARE_REVVERSION="${RUNNING_FIRMWARE_VERSION##0.4.}"
    INSTALLING_FIRMWARE_REVVERSION="${INSTALLING_FIRMWARE_VERSION##0.4.}"
    if (( RUNNING_FIRMWARE_REVVERSION >= 30 ))
    then
      LOG "Will not upgrade vsc firmware from $RUNNING_FIRMWARE_VERSION to $INSTALLING_FIRMWARE_VERSION on DCP-F."
      exit 0
    fi
  fi

  if [ "$RUNNING_FIRMWARE_VERSION" == "$INSTALLING_FIRMWARE_VERSION" ]
  then
    LOG "Running version and installing version are the same: ${RUNNING_FIRMWARE_VERSION}"
    exit 0
  elif [ -f "${INSTALLING_FIRMWARE_FILE}" ]
  then

    LOG "Upgrading the VSC firmware from version: $RUNNING_FIRMWARE_VERSION to: $INSTALLING_FIRMWARE_VERSION."

    if [[ -f "/proc/device-tree/dcpr" ]]; then
      MGMT_TUNNEL_ETH_PATH='config/inbandManagementTunnel/ethernetInterface'
      MGMT_TUNNEL_ETH_IFACE=$(tiny_dh_client -d ${VSC_DH_FILE} -c gets -a ${MGMT_TUNNEL_ETH_PATH})
      if [[ -z ${MGMT_TUNNEL_ETH_IFACE} ]] || [[ ${MGMT_TUNNEL_ETH_IFACE} == 'null' ]]; then
        LOG "Failed to get the inband management tunnel ethernet interface: [${MGMT_TUNNEL_ETH_IFACE}]"
        RUNNING_RELEASE_VERSION=$(cat /etc/image/release.txt)
        if [[ "$RUNNING_RELEASE_VERSION" =~ ^7.[0-9].[0-9] ]]; then
          LOG "Disabling ${MGMT_TUNNEL_ETH_PATH} before upgrading firmware in release: [${RUNNING_RELEASE_VERSION}]"
          MGMT_TUNNEL_ETH_IFACE='disabled'
          tiny_dh_client -d ${VSC_DH_FILE} -c set -a ${MGMT_TUNNEL_ETH_PATH} -t string -v ${MGMT_TUNNEL_ETH_IFACE}
        else
          THROW 1 "Please configure tunnel port before upgrading firmware!"
        fi
      else
        LOG "Inband management tunnel ethernet interface: [${MGMT_TUNNEL_ETH_IFACE}]"
      fi
    fi

    # Slot module running Smartmanager should not do fwupgrade while chassis is fetching data from Smartmanager
    if [[ -f /usr/sbin/smartmanager/smartmanager.js ]] && [[ ! -f "/proc/device-tree/is_chassis" ]]; then
      chassis_ip=$(tiny_dh_client -d /tmp/device_handlers/network -c gets -a data/chassisLinkLocalAddress)
      timer=300
      redis_port=50001
      while true; do
        # Check if redis is up
        status=$(redis-cli -p ${redis_port} -h ${chassis_ip} PING 2>/dev/null)
        if [[ "${status}" == "PONG" ]]; then
          is_migration_running=$(redis-cli -p ${redis_port} -h ${chassis_ip} config get dir | grep migrate)
          if [[ -z ${is_migration_running} ]]; then
            # Not a migration instance
            break
          fi
        else
          # Redis is offline
          break
        fi
        timer=$((timer-1))
        LOG "Waiting for chassis migration redis at: ${chassis_ip} ${redis_port} ... timeout="${timer}
        if [[ ${timer} == 0 ]]; then
          # Timeout
          break
        fi
        sleep 5
      done
    fi

    fwupgrade --url file:"${INSTALLING_FIRMWARE_FILE}" || fail=1

    if [ $fail -ne 0 ]
    then
      LOG "Failed to upgrade the VSC firmware to version: $INSTALLING_FIRMWARE_VERSION."
      exit 1
    else
      LOG "Successfully upgraded the VSC firmware to version: $INSTALLING_FIRMWARE_VERSION."

      # Handle reconfiguration of tunnel interface on slaves after fwupgrade & reset of switch
      if [[ -f "/proc/device-tree/dcpr" ]]; then
        num_iterations=0
        while ((num_iterations++ < ${RECONFIG_TUNNEL_AFTER_FWUPGRADE_RETRY})); do
          CMD_OUTPUT=$(tiny_dh_client -d ${VSC_DH_FILE} -c set -a ${MGMT_TUNNEL_ETH_PATH} -t string -v ${MGMT_TUNNEL_ETH_IFACE})
          ret=$?
          LOG "Set inband management tunnel ethernet interface to ${MGMT_TUNNEL_ETH_IFACE}, output [${CMD_OUTPUT}], iter=${num_iterations}"
          [[ $ret -eq 0 ]] && break
          sleep 1
        done
      fi
      exit 0
    fi

  else
    LOG "VSC firmware file "${INSTALLING_FIRMWARE_FILE}" is not found."
  fi
else
  LOG "VSC device handler is not found."
fi

exit 0
