#!/bin/bash

if [[ ! -f "/proc/device-tree/is_chassis" ]]; then
  LOG "Skip migrate data for this module!"
  exit 0
fi

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
fi

LOG "Migration script for Smartmanager configs, alarms, states starting..."

SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)

# Much be matched with 50001.conf
REDIS_PORT=50001
WORKSPACE="/tmp/migrate/"
AOF_FILE=${WORKSPACE}"/${REDIS_PORT}.aof"
DB_FILE_DESTINATION=${PERSISTENT_DIRECTORY}"/etc/sw_storage/redis/"


mkdir -p ${WORKSPACE}
rm -rf ${WORKSPACE}/*

cp -r ${SCRIPTS_DIR}/* ${WORKSPACE}

JQ_BIN=${WORKSPACE}/jq
export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:"${WORKSPACE}"

CLI_BIN="/usr/bin/redis-cli"
CURL_BIN="/usr/bin/curl"
CURL_ARG="--max-time 30"

# Redis-profile
DEVICE_PRODUCT=$(tr -d '\0' </proc/device-tree/product)
DEVICE_REVISION=$(tr -d '\0' </proc/device-tree/major_revision)
DEVICE_MAC_SHORT=$(cat /sys/class/net/eth0/address |cut -d: -f4- | tr -d :)
SERIAL_NO=$(tiny_dh_client -d /tmp/device_handlers/system_info -a inventory/serialNumber -c gets)
REDIS_CONF_HASH_TAG="{${SERIAL_NO}00C}"
REDIS_DATA_HASH_TAG="{${SERIAL_NO}00D}"

DCP_DEFAULTS_PATH="${WORKSPACE}/default/dcp"

if [ -f /proc/device-tree/is_chassis ]; then
    export DEVICE_IS_CHASSIS=true
else
    unset DEVICE_IS_CHASSIS
fi
if [ -f /proc/device-tree/is_module ]; then
    export DEVICE_IS_MODULE=true
else
    unset DEVICE_IS_MODULE
fi

export DEVICE_PRODUCT DEVICE_REVISION DEVICE_MAC_SHORT
export REDIS_CONF_HASH_TAG REDIS_DATA_HASH_TAG
export DCP_DEFAULTS_PATH

function ping_redis() {
  timer=20
  redis_port=${1}
  is_up=false
  while true; do
      # Check if redis is up
      status=$(${CLI_BIN} -p ${redis_port} PING)
      if [[ "${status}" == 'PONG' ]]; then
          is_up=true
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for redis... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
        is_up=false
        break
      fi
      sleep 1
  done
  echo ${is_up}
}

# Start redis-server
pkill -f redis-server-6 &>/dev/null
sleep 1s
${WORKSPACE}/redis-server-6 ${WORKSPACE}/${REDIS_PORT}.conf &>/dev/null&
PID=$!
sleep 1s

PING_REDIS=`ping_redis ${REDIS_PORT}`
REDIS_VERSION=`${WORKSPACE}/redis-server-6 --version`
if [[ ${PING_REDIS} == true ]]; then
  LOG "Redis is up and running at port: ${REDIS_PORT}, version: ${REDIS_VERSION}"
  sleep 1s # Waiting for some unexpected paths to be loaded
  ${CLI_BIN} -p ${REDIS_PORT} FLUSHALL SYNC
else
  LOG "Failed to start redis server at port: ${REDIS_PORT}, version: ${REDIS_VERSION}"
  pkill -f redis-server-6
  exit 1
fi


# Migrate data

# Path to be put into the DH.CREATE_CONFIG_KEY
# e.g  input: /system/config/tacplus/secondary/port
#     output:         config/tacplus/secondary/port
function resolve_new_path() {
  ORIG_PATH=${1}
  PREFIX=${2}
  NEW_PATH=''

  # Syntax ("pathname-in-R6","changed-name-in-R7"), comma-separated without any spaces
  path_changed=()
  path_changed+=("system/config/syslog/server","system/config/syslog/primaryServer")
  path_changed+=("system/config/syslog/port","system/config/syslog/primaryPort")
  path_changed+=("system/config/syslog/protocol","system/config/syslog/primaryProtocol")
  path_changed+=("snmp/config/trapDestinationAdd","snmp/config/trapDestination")

  for row in ${path_changed[@]}; do
    ORIG_PATH_R6=${row%%,*}
    ORIG_PATH_R7=${row##*,}
    if [[ ${ORIG_PATH} =~ ${ORIG_PATH_R6} ]]; then
      ORIG_PATH=${ORIG_PATH_R7}
      break
    fi
  done

  # Remove the first token: '/'
  [[ ${ORIG_PATH::1} == '/' ]] && ORIG_PATH="${ORIG_PATH:1}"

  # Check that we should remove the prefix or not
  if [[ ${ORIG_PATH} =~ 'interfaceDescription' ]]; then

    if [[ ${ORIG_PATH} =~ ^network/config/(.*)/interfaceDescription ]]; then
      IFACE=${BASH_REMATCH[1]}
      NEW_PATH=${IFACE}"/config/description"
    elif [[ ${ORIG_PATH} =~ ^interface/([[:alnum:]]+)/config/interfaceDescription ]]; then
      IFACE=${BASH_REMATCH[1]}
      NEW_PATH=${IFACE}"/config/description"
    elif [[ ${ORIG_PATH} =~ ^wss_edge/config/wss/(1/|2/|)channel/([[:digit:]]+)/config/interfaceDescription ]]; then
      PORT=${BASH_REMATCH[1]}
      [[ ${PORT} == '' ]] && PORT="1/" # DCP-F-R22
      CHANNEL=${BASH_REMATCH[2]}
      NEW_PATH="wss/"${PORT}${CHANNEL}"/config/description"
    elif [[ ${ORIG_PATH} =~ ^booster/config/interfaceDescription ]]; then
      NEW_PATH="config/description"
    else
      NEW_PATH=${ORIG_PATH}
    fi

  elif [[ ${PREFIX} =~ (dh/traffic_manager) ]]; then
    NEW_PATH=${ORIG_PATH}
  elif [[ ${ORIG_PATH} =~ ^(slot|psu|sfp)/[[:digit:]]+/ ]]; then
    # Remove the device handler prefix which has string and number
    NEW_PATH=`echo ${ORIG_PATH} | cut -d '/' -f3-`
  else
    # Remove the device handler prefix
    NEW_PATH=`echo ${ORIG_PATH} | cut -d '/' -f2-`
  fi

  echo ${NEW_PATH}
}

# List of device handlers that have config attributes and states/alarms
function resolve_device_handler() {
  ORIG_PATH=${1}
  PREFIX=''

  # Generic paths
  if [[ ${ORIG_PATH} =~ /interfaceDescription$ ]] ||
     [[ ${ORIG_PATH} =~ /system/config/(inactivitytimeout|topologyInternal)$ ]]; then

    if [[ ${ORIG_PATH} =~ ^/wss_edge/config/wss/(1/|2/|)channel/([[:digit:]]+)/config/interfaceDescription ]]; then
      PREFIX=so/channel
    else
      # Collect the device handler prefix
      DH_PREFIX=`echo ${ORIG_PATH} | cut -d '/' -f-2`
      [[ ${DH_PREFIX} =~ ^/system ]] && DH_PREFIX='/system/settings'
      PREFIX=so${DH_PREFIX}
    fi

  elif [[ ${ORIG_PATH} =~ ^/system/ ]]; then
    PREFIX="dh/system/settings"
  elif [[ ${ORIG_PATH} =~ ^/fan/ ]]; then
    PREFIX="dh/fan"
  elif [[ ${ORIG_PATH} =~ ^/network/ ]]; then
    PREFIX="dh/network"
  elif [[ ${ORIG_PATH} =~ ^/snmp/ ]]; then
    PREFIX="dh/snmp"
  elif [[ ${ORIG_PATH} =~ ^/((slot|psu)/[[:digit:]])/ ]]; then
    PREFIX="dh/${BASH_REMATCH[1]}"

  # DCP-M
  elif [[ ${ORIG_PATH} =~ ^/cloop/ ]]; then
    PREFIX="dh/cloop"
  elif [[ ${ORIG_PATH} =~ ^/cloopcrx/ ]]; then
    PREFIX="dh/cloopcrx"
  elif [[ ${ORIG_PATH} =~ ^/cloopctx/ ]]; then
    PREFIX="dh/cloopctx"
  elif [[ ${ORIG_PATH} =~ ^/cloopl/ ]]; then
    PREFIX="dh/cloopl"
  elif [[ ${ORIG_PATH} =~ ^/led_dcpm/ ]]; then
    PREFIX="dh/led_dcpm"

  # DCP-101
  elif [[ ${ORIG_PATH} =~ ^/line/ ]]; then
    PREFIX="dh/cfp"
  elif [[ ${ORIG_PATH} =~ ^/gb/ ]]; then
    PREFIX="dh/bcm_gb"
  elif [[ ${ORIG_PATH} =~ ^/client/ ]]; then
    PREFIX="dh/qsfp"

  # DCP-108
  elif [[ ${ORIG_PATH} =~ ^/interface/ ]]; then
    PREFIX="dh/traffic_manager"
  elif [[ ${ORIG_PATH} =~ ^/transponder/ ]]; then
    PREFIX="dh/traffic_manager"

  # DCP-1610
  elif [[ ${ORIG_PATH} =~ ^/fpga/ ]]; then
    PREFIX="dh/fpga"
  elif [[ ${ORIG_PATH} =~ ^/sfp/([[:digit:]]+)/  ]]; then
    SFP_NO=${BASH_REMATCH[1]}
    PREFIX="dh/sfp/${SFP_NO}"

  # DCP-F
  elif [[ ${ORIG_PATH} =~ ^/led_dcpf/ ]]; then
    PREFIX="dh/led_dcpf"
  elif [[ ${ORIG_PATH} =~ ^/booster/ ]]; then
    PREFIX="dh/traffic_manager"
  elif [[ ${ORIG_PATH} =~ ^/control/ ]]; then
    PREFIX="dh/traffic_manager"
  elif [[ ${ORIG_PATH} =~ ^/wss_edge/ ]]; then
    PREFIX="dh/traffic_manager"
  elif [[ ${ORIG_PATH} =~ ^/osc/ ]]; then
    PREFIX="dh/traffic_manager"

  else
    DH_CUT=`echo ${ORIG_PATH} | cut -d '/' -f-2`
    PREFIX="dh"${DH_CUT}
  fi

  echo ${PREFIX}
}

function append_states_to_redis_hash() {
  ORIG_PATH=${1}
  VALUE=${2}
  REDIS_HASH_PATH="mgr/alarm:alarms"
  MY_VALUE=''
  if [[ ${VALUE} =~ '"state":"activated"' ]]; then
    MY_VALUE='{"data":{"severity":1,"previous":0,"activated":0,"raised":0},"state":"Activated"}'
  else
    MY_VALUE='{"data":{"severity":0,"previous":0,"activated":0,"raised":0},"state":"Idle"}'
  fi

  LOCATION='/'
  if [[ ${ORIG_PATH} =~ '/module/1/' ]]; then
    LOCATION='/module/1/'
    ORIG_PATH="${ORIG_PATH:9}"
  elif [[ ${ORIG_PATH} =~ '/module/2/' ]]; then
    LOCATION='/module/2/'
    ORIG_PATH="${ORIG_PATH:9}"
  fi

  DH_PREFIX=`resolve_device_handler ${ORIG_PATH}`
  PATH_REMAIN=`resolve_new_path ${ORIG_PATH} ${DH_PREFIX}`

  NEW_PATH=${DH_PREFIX}:${PATH_REMAIN}
  LOG "Path for state: ${NEW_PATH} Value: ${MY_VALUE}"

  ${CLI_BIN} -p ${REDIS_PORT} eval "
    local states_obj = \"{}\";
    local hash_key = '${REDIS_CONF_HASH_TAG}:${REDIS_HASH_PATH}'
    local hash_field = '${LOCATION}'
    local redis_read = redis.call('hget', hash_key, hash_field);
    if redis_read then states_obj = redis_read end;
    local json_obj = cjson.decode(states_obj);
    local data = cjson.decode('${MY_VALUE}');
    json_obj['${NEW_PATH}'] = data ;
    redis.call('hset', hash_key, hash_field, cjson.encode(json_obj));
    " 0
}

# Make sure slot modules are in good state
if [[ -f "/proc/device-tree/dcp2" ]]; then
  traffic_module_dh=("" "slotA" "slotB")
  timer=30
  while true; do
    success=true
    for slot_id in {1,2}; do
      traffic_module_dh_path="/tmp/device_handlers/traffic_module/${traffic_module_dh[$slot_id]}"
      module_present=$(tiny_dh_client   -d $traffic_module_dh_path -c gets -a present)
      address_detected=$(tiny_dh_client -d $traffic_module_dh_path -c gets -a addressDetected)
      if [[ "${module_present}" == "true" && "${address_detected}" == "true" ]]; then
        module_status=$(${CURL_BIN} ${CURL_ARG} localhost:8080/status/module/${slot_id} 2>/dev/null | ${JQ_BIN} -r '.value')
        if [[ ${module_status} != "ok" ]]; then
          LOG "Slot module $slot_id is not ready: ${module_status} timeout=${timer}"
          success=false
          break
        else
          LOG "Slot module $slot_id is ready"
        fi
      fi
    done
    if [[ $success == false ]]; then
      timer=$((timer-1))
      if [[ ${timer} == 0 ]]; then
        LOG "Waiting for module $slot_id: timeout!"
        pkill -f redis-server-6
        exit 1
      fi
      sleep 1
      continue
    fi
    break
  done
fi

LOG "### Start migrating data for Smartmanager alarms ###"
ALARM_LOG=`${CURL_BIN} ${CURL_ARG} localhost:8080/v1/alarms/log?sort=id 2>/dev/null`
CURL_ERR=$?
if [[ ${CURL_ERR} == 0 ]]; then
  ALARM_ENTRY=""
  CURRENT_PATH=""
  CURRENT_STARTED=""
  CURRENT_STOPPED=""
  BRACKET_OPEN=""
  ACTIVATED_STATE='"state":"activated"'

  # Iterate the alarm log to convert paths to new format and add alarm running state
  while read -r line; do
    if [[ ${line} =~ "{" ]]; then
      ALARM_ENTRY=${line}
      BRACKET_OPEN="true"
      CURRENT_PATH=""
      CURRENT_STARTED=""
      CURRENT_STOPPED=""
      CURRENT_LOCATION=""
    elif [[ ${line} =~ "}" ]]; then
      if [[ ! -z ${BRACKET_OPEN} ]]; then
        ALARM_ENTRY=${ALARM_ENTRY}"}"
        LOG "Add alarm entry: ${ALARM_ENTRY}"
        redis-cli -p ${REDIS_PORT} XADD ${REDIS_CONF_HASH_TAG}:mgr/alarm:log \* data "${ALARM_ENTRY}" version "1.0"

        # Do not have 'stopped' entry, it means running state is activated
        if [[ ! -z ${CURRENT_STARTED} ]] && [[ -z ${CURRENT_STOPPED} ]]; then
          COMBINED_PATH=${CURRENT_PATH}
          [[ ${COMBINED_PATH::1} == '/' ]] && COMBINED_PATH="${COMBINED_PATH:1}"
          COMBINED_PATH="${CURRENT_LOCATION}${COMBINED_PATH}"
          LOG "Added default running state for combined path: ${COMBINED_PATH}"
          append_states_to_redis_hash ${COMBINED_PATH} ${ACTIVATED_STATE}
        fi

        BRACKET_OPEN=""
        ALARM_ENTRY=""
      else
        ALARM_ENTRY=${ALARM_ENTRY}"${line}"
      fi
    elif [[ ${line} =~ \"path\":[[:space:]]*\"(.*)\" ]]; then
      CURRENT_PATH="${BASH_REMATCH[1]}"
      DH_PREFIX=`resolve_device_handler ${CURRENT_PATH}`
      PATH_REMAIN=`resolve_new_path ${CURRENT_PATH} ${DH_PREFIX}`
      ALARM_ENTRY=${ALARM_ENTRY}"\"path\": \"${DH_PREFIX}:${PATH_REMAIN}\","
    elif [[ ${line} =~ \"started\":[[:space:]]*\"(.*)\" ]]; then
      ALARM_ENTRY=${ALARM_ENTRY}"${line}"
      # Extra fields are required by Alarm manager: start_ts, stop_ts
      ALARM_ENTRY=${ALARM_ENTRY}"\"start_ts\":0,\"stop_ts\":0,"
      CURRENT_STARTED="${BASH_REMATCH[1]}"
    elif [[ ${line} =~ \"stopped\":[[:space:]]*\"(.*)\" ]]; then
      ALARM_ENTRY=${ALARM_ENTRY}"${line}"
      CURRENT_STOPPED="${BASH_REMATCH[1]}"
    elif [[ ${line} =~ \"location\":[[:space:]]*\"(.*)\" ]]; then
      ALARM_ENTRY=${ALARM_ENTRY}"${line}"
      CURRENT_LOCATION="${BASH_REMATCH[1]}"
    else
      ALARM_ENTRY=${ALARM_ENTRY}"${line}"
    fi
  done < <(echo "${ALARM_LOG}")

else # curl failed
  LOG "Failed to read alarm log from Smartmanager: /v1/alarms/log"
  pkill -f redis-server-6
  exit 1
fi

LOG "### Start migrating data for Smartmanager states ###"
PING_REDIS=`ping_redis 6379`
if [[ ${PING_REDIS} == true ]]; then
  ${CLI_BIN} keys /v1/states/* | xargs -I{} sh -c "echo -n '{} '; ${CLI_BIN} -c get {};" | while read -r entry; do
    ORIG_PATH=`eval echo ${entry} | cut -d ' ' -f1`
    VALUE=`echo ${entry} | cut -d ' ' -f2-`
    [[ ${ORIG_PATH} =~ '/v1/states/' ]] && ORIG_PATH="${ORIG_PATH:10}"
    append_states_to_redis_hash ${ORIG_PATH} ${VALUE}
  done
else
  LOG "Failed to read state from local redis port 6379"
  SM_STATES="/var/persistent/etc/smartmanager/states.json"
  LOG "Trying with ${SM_STATES}"
  if [[ -f ${SM_STATES} ]]; then
    ${JQ_BIN} -c '.' ${SM_STATES} | while read -r entry; do
      ORIG_PATH=`echo ${entry}  | ${JQ_BIN} -r -c '.path'`
      VALUE=${entry}
      append_states_to_redis_hash ${ORIG_PATH} ${VALUE}
    done
  else
    LOG "File not found: ${SM_STATES}"
  fi
fi


LOG "### Start migrating data for config.json ###"

# Trigger Smartmanager to flush all resetable config data to disk
# FLUSH2DISK=`${CURL_BIN} ${CURL_ARG} -X POST -H "Content-Type: application/json" -d '{"value":true}' localhost:8080/v1/config/flush2disk 2>/dev/null`
# LOG "FLUSH2DISK: ${FLUSH2DISK} code: $?"
# sleep 3s # Smartmanager FLUSH_TIMEOUT = 1000ms

# Hostname migration
HOSTNAME=$(/bin/hostname)
LOG "Migrate hostname Value: ${HOSTNAME}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/system/settings config/hostname "{\"value\":\"${HOSTNAME}\"}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/snmp            config/sysname  "{\"value\":\"${HOSTNAME}\"}"

SM_CONFIG="/tmp/config.json"
# Copy to /tmp to be able to add missing config
cp /var/persistent/etc/smartmanager/config.json ${SM_CONFIG}

# DC-3223 New Platform: Procedure to keep config when upgrade to R7.0
# Some config will be lost if upgraded from 5.3 or earlier.
declare -a missing_paths=("/system/config/syslog/access"
                          "/system/config/syslog/adminStatus"
                          "/system/config/syslog/alarm"
                          "/system/config/syslog/config"
                          "/system/config/syslog/port"
                          "/system/config/syslog/protocol"
                          "/system/config/syslog/server"
                          "/system/config/tacplus/primary/key"
                          "/system/config/tacplus/primary/address"
                          "/system/config/tacplus/adminStatus"
                          "/system/config/tacplus/secondary/address"
                          "/system/config/tacplus/secondary/key"
                          "/system/config/tacplus/retry"
                          "/system/config/tacplus/timeout"
                          "/system/config/radius/timeout"
                          "/system/config/radius/secondary/address"
                          "/system/config/radius/secondary/key"
                          "/system/config/radius/primary/address"
                          "/system/config/radius/primary/key"
                          "/system/config/radius/retry"
                          "/system/config/radius/secondary/port"
                          "/system/config/radius/primary/port"
                          "/system/config/radius/adminStatus"
                          )
if [[ -f /proc/device-tree/dcpm ]]; then
  AUTOMATIONMODE_FILE="/var/persistent/etc/automation-mode-startup.conf"
  VALUE=$(cat ${AUTOMATIONMODE_FILE} | grep automationMode | awk -F ' ' '{print $2}')
  LOG "Orig: /cloop/config/automationMode Prefix: dh/cloop Path: config/automationMode Value: ${VALUE}"
  ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/cloop config/automationMode "{\"value\":\"${VALUE}\"}"

  SINGLEFIBERMODE_CONF_FILE="/var/persistent/etc/single-fiber-mode.conf"
  SINGLEFIBER_MODE=$(<$SINGLEFIBERMODE_CONF_FILE)
  if [[ ! -z ${SINGLEFIBER_MODE} ]]; then
    LOG "Creating config Prefix: /cloop/config/system/fiberMode Prefix: dh/cloop Path: config/system/fiberMode Value: ${SINGLEFIBER_MODE}"
    ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/cloop config/system/fiberMode "{\"value\":\"${SINGLEFIBER_MODE}\"}"
  fi

  GLOBAL_TARGET_OUTPUT_POWER=$(cat ${AUTOMATIONMODE_FILE} | grep targetOutputPower_0_ch | awk -F ' ' '{print $2}')
  if [[ ! -z ${GLOBAL_TARGET_OUTPUT_POWER} ]]; then
    LOG "Orig: /cloopcrx/config/system/setGlobalTargetOutputPower Prefix: dh/cloopcrx Path: config/system/setGlobalTargetOutputPower Value: ${GLOBAL_TARGET_OUTPUT_POWER}"
    ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/cloopcrx config/system/setGlobalTargetOutputPower "{\"value\":${GLOBAL_TARGET_OUTPUT_POWER}}"
  fi

  grep 'opticalControlMode' ${AUTOMATIONMODE_FILE} | while read -r entry; do
    if [[ ${entry} =~ ^opticalControlMode_(.*)_ch ]]; then
      CHANNEL_NO=${BASH_REMATCH[1]}
      CHANNEL_NO=$((CHANNEL_NO+1))
      MODE_VALUE=$(echo ${entry} | awk -F ' ' '{print $2}')
      REDIS_PATH="config/client/${CHANNEL_NO}/rx/opticalControlMode"
      LOG "Orig: ${entry} Prefix: dh/cloopcrx Path: ${REDIS_PATH} Value: ${MODE_VALUE}"
      ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/cloopcrx ${REDIS_PATH} "{\"value\":\"${MODE_VALUE}\"}"
    fi
  done

  missing_paths+=("/cloopcrx/config/system/pamMode")
  missing_paths+=("/cloopl/config/system/pamMode")
  OPTICAL_FILE="/var/persistent/etc/optical-parameters.conf"
  if [[ -f ${OPTICAL_FILE} ]]; then
    PAM_MODE=`grep "optical_pam4_mode" ${OPTICAL_FILE} | awk '{print $2 }' | xargs | tr '[:upper:]' '[:lower:]'`
    if [[ ${PAM_MODE} == 'manual' ]]; then
      missing_paths+=("/cloopcrx/config/system/targetPower")
      missing_paths+=("/cloopl/config/system/targetPower")
      LOG "PAM mode manual, save targetPower!"
    fi
  fi
fi

# Tacacs default port is wrong at -1, we should read it from config file instead of SM
TACACS_CONF_FILE="/var/persistent/etc/tacplus_servers"

PRIMARY_PORT=$(cat ${TACACS_CONF_FILE} | grep "server=" | head -n1 | sed -n 's/^server=.*:\(.*\)/\1/p')
if [[ ! -z ${PRIMARY_PORT} ]]; then
  LOG "Creating config Prefix: dh/system/settings Path: config/tacplus/primary/port Value: ${PRIMARY_PORT}"
  ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/system/settings config/tacplus/primary/port "{\"value\":${PRIMARY_PORT}}"
fi

SECONDARY_PORT=$(cat ${TACACS_CONF_FILE} | grep "server=" | tail -n1 | sed -n 's/^server=.*:\(.*\)/\1/p')
if [[ ! -z ${SECONDARY_PORT} ]]; then
  LOG "Creating config Prefix: dh/system/settings Path: config/tacplus/secondary/port Value: ${SECONDARY_PORT}"
  ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/system/settings config/tacplus/secondary/port "{\"value\":${SECONDARY_PORT}}"
fi

echo "" >> ${SM_CONFIG}
for missing_path in "${missing_paths[@]}"; do
   config_value=`${CURL_BIN} ${CURL_ARG} localhost:8080/v1/rows${missing_path} 2>/dev/null | ${JQ_BIN} -c --arg path_add $missing_path '{"path":$path_add} + .'`
   LOG "Fetch config from SM: ${config_value}"
   echo ${config_value} >> ${SM_CONFIG}
done

if [[ -f ${SM_CONFIG} ]]; then

  declare -A PATH_VALUE
  arr_index=0
  ${JQ_BIN} '.path,.value,.time' ${SM_CONFIG} | while read -r entry; do
    PATH_VALUE[${arr_index}]=${entry}
    if [[ ${arr_index} == 2 ]]; then
      arr_index=0
      ORIG_PATH=`eval echo ${PATH_VALUE[0]}`
      VALUE=${PATH_VALUE[1]}
      TIMESTAMP=${PATH_VALUE[2]}

      if [[ ${ORIG_PATH} =~ ^/system/settings/ ]]; then
        LOG "Skip this path: ${ORIG_PATH}"
        continue;
      fi

      DH_PREFIX=`resolve_device_handler ${ORIG_PATH}`
      NEW_PATH=`resolve_new_path ${ORIG_PATH} ${DH_PREFIX}`
      [[ ${NEW_PATH::1} == '/' ]] && NEW_PATH="${NEW_PATH:1}"

      LOG "Orig: ${ORIG_PATH} Prefix: ${DH_PREFIX} Path: ${NEW_PATH} Value: ${VALUE} Timestamp: ${TIMESTAMP}"
      ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} ${DH_PREFIX} ${NEW_PATH} "{\"value\":${VALUE}}"
      if [[ ! "${TIMESTAMP}" == "null" ]]; then
        REDIS_FULL_PATH="${REDIS_CONF_HASH_TAG}:${DH_PREFIX}:${NEW_PATH}"
        ${CLI_BIN} -p ${REDIS_PORT} DH.SET_OPTION ${REDIS_FULL_PATH} CREATED_AT ${TIMESTAMP} MODIFIED_AT ${TIMESTAMP}
      fi
    else
      arr_index=$((arr_index+1))
    fi
  done
else
  LOG "File not found: ${SM_CONFIG}"
fi

function resolve_module_conf_hash_tag() {
  SLOT_ID=${1}
  SERIAL_NO=$(redis-cli get /v1/rows/module/${SLOT_ID}/system/inventory/serialNumber | ${JQ_BIN} -r '.value')
  CONF_HASH_TAG="{${SERIAL_NO}00C}"

  echo ${CONF_HASH_TAG}
}

# Handle config data for slot modules
if [[ -f "/proc/device-tree/dcp2" ]]; then
  for slot_id in {1,2}; do
    STATUS=$(${CURL_BIN} ${CURL_ARG} localhost:8080/status/module/${slot_id} 2>/dev/null | ${JQ_BIN} -r '.value')
    if [[ ${STATUS} != "ok" && ${STATUS} != "sw incompatible" ]]; then
      LOG "Module ${slot_id} not present, skip upgrade configurations for this slot"
      # Without module present, we can not detect product name, mac address which are needed to build CONF_HASH_TAG
      continue
    fi

    SLOT_CONF_HASH_TAG=`resolve_module_conf_hash_tag ${slot_id}`
    LOG "Slot ${slot_id} CONF HASH TAG: ${SLOT_CONF_HASH_TAG}"

    SM_CONFIG_SLOT="/var/persistent/etc/smartmanager/module-${slot_id}.json"
    if [[ -f ${SM_CONFIG_SLOT} ]]; then
      LOG "### Start migrating data for module-${slot_id}.json ###"

      declare -A PATH_VALUE
      arr_index=0
      ${JQ_BIN} -c '.config.rows[] | .path,.value' ${SM_CONFIG_SLOT} | while read -r entry; do
        PATH_VALUE[${arr_index}]=${entry}
        if [[ ${arr_index} == 1 ]]; then
          arr_index=0
          ORIG_PATH=`eval echo ${PATH_VALUE[0]}`
          VALUE=${PATH_VALUE[1]}

          DH_PREFIX=`resolve_device_handler ${ORIG_PATH}`
          NEW_PATH=`resolve_new_path ${ORIG_PATH} ${DH_PREFIX}`
          [[ ${NEW_PATH::1} == '/' ]] && NEW_PATH="${NEW_PATH:1}"

          LOG "Slot ${slot_id} Orig: ${ORIG_PATH} Prefix: ${DH_PREFIX} Path: ${NEW_PATH} Value: ${VALUE}"
          ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${SLOT_CONF_HASH_TAG} ${DH_PREFIX} ${NEW_PATH} "{\"value\":${VALUE}}"
        else
          arr_index=$((arr_index+1))
        fi
      done
    else
      LOG "File not found: ${SM_CONFIG_SLOT}"
    fi

    # DCP-F control loop stores config values in traffic-manager.conf in the slot module
    # DC-5550 DCP-F*: Upgrade to R7.1.1 sets wanted output power to (4) default value.
    declare -a missing_slot_paths=("/control/config/maxChannels")
    MODULE_PRODUCT_NAME=$(redis-cli get /v1/rows/module/${slot_id}/system/inventory/productName | ${JQ_BIN} -r '.value')
    if [[ ${MODULE_PRODUCT_NAME} =~ 'DCP-F-R22' ]]; then
      LOG "### Start migrating missing config value for slot ${slot_id}"
       declare -a missing_slot_paths=("/control/config/maxChannels"
                          "/control/config/edfa/defaultAttenuation"
                          "/control/config/express/defaultAttenuation"
                         )
      for channel_no in {1..48}; do
        missing_slot_paths+=("/control/config/${channel_no}/wantedPower")
      done
    elif [[ ${MODULE_PRODUCT_NAME} =~ 'DCP-F-DE22' ]]; then
      LOG "### Start migrating missing config value for slot ${slot_id}"
       declare -a  missing_slot_paths=("/control/config/combinedMode"
                          "/control/config/edfa/defaultAttenuation"
                          "/control/config/eq/defaultAttenuation"
                          "/control/config/eq/ocmPowerOffset"
                         )
      for port_no in {1..2}; do
        missing_slot_paths+=("/control/config/port/${port_no}/maxChannels")
        for channel_no in {1..48}; do
          missing_slot_paths+=("/control/config/port/${port_no}/channel/${channel_no}/defaultAttenuation")
          missing_slot_paths+=("/control/config/port/${port_no}/channel/${channel_no}/wantedPower")
        done
      done
    fi
    for missing_slot_path in "${missing_slot_paths[@]}"; do
       config_value=`${CURL_BIN} ${CURL_ARG} localhost:8080/v1/rows/module/${slot_id}${missing_slot_path} 2>/dev/null`
       CURL_ERR=$?
       if [[ ${CURL_ERR} == 0 ]]; then
          if [[ ${config_value} =~ '(TreeDatabase) Could not find' ]] ; then
            LOG "### Slot ${slot_id} path not found: ${missing_slot_path}"
            continue;
          fi

          if [[ ${config_value} =~ '{"value":null}' ]] ; then
            sibling_data_path=$(echo "${missing_slot_path/\/config\//\/data\/}")
            LOG "### Slot ${slot_id} path has not configured: ${missing_slot_path} = ${config_value}, try read sibling data path: ${sibling_data_path}"
            config_value=`${CURL_BIN} ${CURL_ARG} localhost:8080/v1/rows/module/${slot_id}${sibling_data_path} 2>/dev/null`
            CURL_ERR=$?
            if [[ ${CURL_ERR} == 0 ]]; then
              LOG "### Slot ${slot_id} found sibling data: ${sibling_data_path} = ${config_value}"
            else # curl failed
              LOG "Slot ${slot_id} Failed to fetch sibling data path ${sibling_data_path} from Smartmanager!"
              continue;
            fi
          fi

          ORIG_PATH=${missing_slot_path}
          VALUE=${config_value}
          DH_PREFIX=`resolve_device_handler ${ORIG_PATH}`
          NEW_PATH=`resolve_new_path ${ORIG_PATH} ${DH_PREFIX}`
          [[ ${NEW_PATH::1} == '/' ]] && NEW_PATH="${NEW_PATH:1}"
          LOG "Slot ${slot_id} Orig: ${ORIG_PATH} Prefix: ${DH_PREFIX} Path: ${NEW_PATH} Value: ${VALUE}"
          ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${SLOT_CONF_HASH_TAG} ${DH_PREFIX} ${NEW_PATH} ${VALUE}
       else # curl failed
          LOG "Slot ${slot_id} Failed to fetch path ${missing_slot_path} from Smartmanager!"
       fi
    done
  done

fi

LOG "Handle converting network configuration from interfaces file to redis config keys"
# No need to handle DCP-R since there is no DCP-R for release before R7
br0_ifconfig=$(ifconfig br0 | grep "inet addr" | xargs)
mgmt_ip=$(echo ${br0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
mgmt_netmask=$(echo ${br0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
mgmt_gateway=$(ip route show br0 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
LOG "Collected br0 network settings: [${mgmt_ip}] [${mgmt_netmask}] [${mgmt_gateway}]"
mgmt_config="{\"value\":{\"ipAddress\":\"${mgmt_ip}\",\"netmask\":\"${mgmt_netmask}\",\"gateway\":\"${mgmt_gateway}\"}}"
LOG "Creating config Prefix: dh/network Path: config/br0/ipv4Settings Value: ${mgmt_config}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/network config/br0/ipv4Settings ${mgmt_config}

# eth0 does not have any gateway configured
eth0_ifconfig=$(ifconfig eth0 | grep "inet addr" | xargs)
eth0_ipaddr=$(echo ${eth0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
eth0_netmask=$(echo ${eth0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
LOG "Collected eth0 network settings: [${eth0_ipaddr}] [${eth0_netmask}] "
eth0_config="{\"value\":{\"ipAddress\":\"${eth0_ipaddr}\",\"netmask\":\"${eth0_netmask}\",\"gateway\":\"\"}}"
LOG "Creating config Prefix: dh/network Path: config/eth0/ipv4Settings Value: ${eth0_config}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/network config/eth0/ipv4Settings ${eth0_config}

timestamp=`/bin/date`
${CLI_BIN} -p ${REDIS_PORT} set ${REDIS_CONF_HASH_TAG}:so:upgrade/timestamp "{\"value\":\"${timestamp}\"}"

# Handle shadow file for R6
user_hashes='{}'
for user_name in admin; do
  LOG "Migrating shadow hash to redis for user: $user_name"
  user_hash=$(getent shadow $user_name | tr -d '\n')
  nested_object=$(${JQ_BIN} -n --arg key "$user_name" --arg hash "$user_hash" '{($key): {"hash": $hash}}')
  user_hashes=$(echo "$user_hashes" "$nested_object" | ${JQ_BIN} -s '.[0] * .[1]')
done
json_flat=$(echo "$user_hashes" | ${JQ_BIN} -c -r)
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/system/settings config/cliUsers "{\"value\":$json_flat}"

${CLI_BIN} -p ${REDIS_PORT} BGREWRITEAOF

timer=30
while true; do
    # Check if the write operation is finished
    status=$(${CLI_BIN} -p ${REDIS_PORT} info | grep -i aof_rewrite_in_progress)
    if [[ ${status} =~ ':0' ]]; then
        LOG "Redis BGREWRITEAOF finished!"
        break
    fi
    timer=$((timer-1))
    LOG "Waiting for BGREWRITEAOF... timeout="${timer}
    if [[ ${timer} == 0 ]]; then
      LOG "Failed to save AOF file!"
      pkill -f redis-server-6
      exit 1
    fi
    sleep 1
done


# Store the redis AOF file to persistent directory
fail=0
if [ -f "${AOF_FILE}" ]; then
  LOG "Redis AOF file created: "$AOF_FILE
  mkdir -p ${DB_FILE_DESTINATION}
  cp ${AOF_FILE} ${DB_FILE_DESTINATION} || fail=1
  if [ $fail -ne 0 ]
  then
    LOG "Failed to store the redis AOF file."
    pkill -f redis-server-6
    exit 1
  else
    LOG "Redis AOF file is stored successfully to: ${DB_FILE_DESTINATION}"
  fi
  sync
else
  LOG "Redis AOF file not found: "$AOF_FILE
fi


# Shutdown redis-server
${CLI_BIN} -p ${REDIS_PORT} SHUTDOWN SAVE

# Cleanup workspace
rm -rf ${WORKSPACE} &>/dev/null

exit 0
