#!/bin/bash

if [[ ! -f "/proc/device-tree/is_chassis" ]]; then
  exit 0
fi

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

[ -z "${REDIS_PORT}" ] && REDIS_PORT=30001

# Detect the present of shelf controller in DCP-R or DCP-2 cluster
if [[ -f "/proc/device-tree/dcpr" ]] || [[ -f "/proc/device-tree/dcp2" ]]; then
  ip_controller=$(redis-cli -p 50002 get "${REDIS_DATA_HASH_TAG}":so:mounts | jq -r '.[] | to_entries | .[] | select(.value.status=="Master" and .value.instance_type=="CONF" and .value.node_type=="Controller") | .value.ip')
  if [[ ! -z "${ip_controller}" ]]; then
    LOG "Found controller at ip: ${ip_controller}"
    REDIS_PORT=50001
  fi
fi

# Will migrate shadow file only if the redis key does not exist
exist=$(redis-cli -c -p ${REDIS_PORT} DH.GET $REDIS_CONF_HASH_TAG:dh/system/settings:config/cliUsers)
if [[ -z $exist ]]; then
  user_hashes='{}'
  for user_name in admin sftpuser; do
    user_hash=$(getent shadow $user_name | tr -d '\n')
    if [[ ! -z $user_hash ]]; then
      LOG "Migrating shadow hash to redis for user: $user_name"
      nested_object=$(jq -n --arg key "$user_name" --arg hash "$user_hash" '{($key): {"hash": $hash}}')
      user_hashes=$(echo "$user_hashes" "$nested_object" | jq -s '.[0] * .[1]')
    fi
  done
  json_flat=$(echo "$user_hashes" | jq -c -r)
  redis-cli -c -p ${REDIS_PORT} eval 'return redis.call("DH.CREATE_CONFIG_KEY", KEYS[1], "dh/system/settings", "config/cliUsers", ARGV[1])' 1 ${REDIS_CONF_HASH_TAG} "{\"value\":$json_flat}"
fi
