#!/bin/bash

if [[ ! -f "/proc/device-tree/dcpr" ]]; then
  exit 0
fi

SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)

function check_correct_master() {
  LOG "Getting SC status"
  sc_role=$(redis-cli -p 50002 get "${REDIS_DATA_HASH_TAG}":so:mounts | jq -r '.[] | to_entries | .[] | select(.value.instance_type=="CONF" and .value.node_type=="Controller") | .value.status')
  LOG "SC role is " "${sc_role}" " according to data instance (this is empty if no SC exists in data mount table)"
  if test "$sc_role"; then
    LOG "Check role of SC in mount table. Role of SC: " "${sc_role}"
    if [[ "${sc_role}" != "Master" ]]; then
      LOG "SC exists in mount table but is not master. SC role is " "${sc_role}" " Aborting with error."
      THROW 1 "Shelf-controller must be master. Not possible to upgrade."
    fi
  fi
  if test -z "$sc_role"; then
    sc_in_conf=$(redis-cli -c -p 50001 get "${REDIS_CONF_HASH_TAG}":so:mounts | jq -r '.[] | to_entries | .[] | select(.value.node_type=="Controller") | .value.role')
    LOG "Checking if SC exists in mount table and if it exists it should be master. Role of SC: " "${sc_in_conf}"
    if test "$sc_in_conf"; then
      LOG "SC configured to master. Aborting with error."
      THROW 1 "Shelf-controller must be master. Not possible to upgrade."
    fi
  fi
}

LOG "Running CLOOP flexgrid upgrade script. This script will convert old cloop channels to new cloop flexgrid channels."

ip_controller=$(redis-cli -p 50002 get "${REDIS_DATA_HASH_TAG}":so:mounts | jq -r '.[] | to_entries | .[] | select(.value.status=="Master" and .value.instance_type=="CONF" and .value.node_type=="Controller") | .value.ip')
LOG "Got IP address to master" "${ip_controller}"

if test -z "$ip_controller"; then
  set_options="-c -p 30001"
else
  set_options="-h ${ip_controller} -c -p 50001"
fi

# Check that if SC exists in mount table it is master
check_correct_master

product_name=$(redis-cli -c -p 50002 dh.get "${REDIS_DATA_HASH_TAG}":dh/system/info:inventory/productName | jq -r '.value')

if test -z "$product_name"; then
  LOG "Failed to get product name."
  THROW 1 "Failed to get product name. Not possible to upgrade."
fi

if [[ ! -f "${SCRIPTS_DIR}/upgrade_channel_plan.lua" ]]; then
  LOG "Script ${SCRIPTS_DIR}/upgrade_channel_plan.lua does not exist. Aborting."
  THROW 1 "Cannot find upgrade script Aborting."
fi

result=$(redis-cli ${set_options} --eval ${SCRIPTS_DIR}/upgrade_channel_plan.lua "${REDIS_CONF_HASH_TAG}" , ${product_name})
# Result can be either 'Failed: <error message>' or 'success: <success message>'.
# Catch ERR* as well, since that is what lua uses to report syntax errors.
if [[ $result == Failed:* || $result == ERR* ]]; then
  error_message="${result#Failed: }"
  LOG "Failed to upgrade channel plan: ${error_message}"
  THROW 1 "Failed to upgrade channel plan: ${error_message}"
fi

if [[ $result != Success:* ]]; then
  LOG "Upgrade did not succeed: ${result}"
  THROW 1 "Upgrade did not succeed: Unexpected error."
fi

check_correct_master
LOG "${result}"

exit 0