-- KEYS[1] = M32 redis tag

local tag = KEYS[1]
local number_of_channels = 32
local old_paths = {}
local new_paths = {}

-- %s is the channel number, 1 to 32. Cluster prefix has to be prepended
-- tx attenuation old to new
old_paths[1] = ":dh/cloopctx:config/client/%s/tx/attenuation"
new_paths[1] = ":dh/cloop:config/client/%s/tx/defaultAttenuation"

-- rx attenuation old to new
old_paths[2] = ":dh/cloopcrx:config/client/%s/rx/defaultAttenuation"
new_paths[2] = ":dh/cloop:config/client/%s/rx/defaultAttenuation"

-- tx adminStatus old to new
old_paths[3] = ":dh/cloopctx:config/client/%s/tx/adminStatus"
new_paths[3] = ":dh/cloop:config/client/%s/tx/adminStatus"

-- rx adminStatus old to new
old_paths[4] = ":dh/cloopcrx:config/client/%s/rx/adminStatus"
new_paths[4] = ":dh/cloop:config/client/%s/rx/adminStatus"

--[[
    The array new_paths_surgery could replace the array new_paths. But it would be less readable.
    The hope is that the reviewer along with any future maintainers will be able to take a look at 
    the old_paths array and instantly compare it to new_paths. Unfortunately, the redis
    command DH.CREATE_CONFIG_KEY demands the path to passed in as three separate arguments.
    Hence the need for the new_paths_surgery array. 
--]]
local new_paths_surgery = {}
new_paths_surgery[1] = {
    "dh/cloop",
    "config/client/%s/tx/defaultAttenuation"
}
new_paths_surgery[2] = {
    "dh/cloop",
    "config/client/%s/rx/defaultAttenuation"
}
new_paths_surgery[3] = {
    "dh/cloop",
    "config/client/%s/tx/adminStatus"
}
new_paths_surgery[4] = {
    "dh/cloop",
    "config/client/%s/rx/adminStatus"
}

local function get_txrx_channel(txrx, channel)
    local key = string.format("%s:dh/cloopc%s:config/client/%s/%s/attenuation", tag, txrx, channel, txrx)
    local result = redis.pcall('dh.get', key)
    result = key .. " = " .. result
    return result
end

local function set_value_to_new_path(json_object, path, device_handler, attribute)
    local surgical_path = tag .. ":" .. device_handler .. ":" .. attribute
    if surgical_path ~= path then -- This is a sanity check
        return { err = surgical_path .. " != " .. path }
    end
    local output = redis.pcall('DH.CREATE_CONFIG_KEY', tag, device_handler, attribute, json_object)
    if type(output) == "table" and output.err then
        return { err = "Failed: Failed to create key for " .. path }
      end
    output = redis.pcall('DH.SET', path, json_object)
    if type(output) == "table" and output.err then
        return { err = "Failed: Failed to set value for " .. path }
    end
    return { err = nil }
end

local function do_upgrade_channel(channel)
    local result = ""
    for i = 1, #old_paths do
        local old_key = tag .. string.format(old_paths[i], channel)
        local new_key = tag .. string.format(new_paths[i], channel)
        local device_handler = new_paths_surgery[i][1]                      -- Basically "dh/cloop", see path surgery comments
        local attribute = string.format(new_paths_surgery[i][2], channel)   -- Basically "config/client/%s/tx/defaultAttenuation", see path surgery comments
        local json_object = redis.pcall('dh.get', old_key)                  -- The thing that has to be moved from old to new path
        if not json_object then
            result = "Error: Failed to fetch old key " .. old_key .. " - json_object is nil\n"
            break
        end
        if json_object.err then
            result = "Error: " .. json_object.err .. "\n"
            break
        end
        local set_result = set_value_to_new_path(json_object, new_key, device_handler, attribute)
        if not set_result then
            result = "Error: Failed to set value for new key " .. new_key .. " - set_result is nil\n"
            break
        end
        if set_result.err then
            result = "Error: " .. set_result.err .. "\n"
            break
        else
            result = result .. "Success: Updated " .. old_key .. " to " .. new_key .. "\n"
            redis.pcall('DEL', old_key) -- Delete old_key
        end
    end
    return result
end

local function do_upgrade_interface(channel)
    --[[
    Because the description is stored in the interface attribute rather than the cloop attribute
    it was indexed using the 9140 format, rather than the channel number.
    There was no neat way of putting it into the do_upgrade_channel() method so it got its own method.
    --]]
    local interface = 9140 + 15 * (channel - 1)
    local old_key = tag .. string.format(":so/interface:%s/config/description", interface)
    local new_key = tag .. string.format(":dh/cloop:config/client/%s/rx/description", channel)
    local device_handler = "dh/cloop"
    local attribute = string.format("config/client/%s/rx/description", channel)
    local json_object = redis.pcall('dh.get', old_key)
    if not json_object then
        return "Error: Failed to fetch old key " .. old_key .. " - json_object is nil\n"
    end
    if json_object.err then
        return "Error: " .. json_object.err .. "\n"
    end
    local set_result = set_value_to_new_path(json_object, new_key, device_handler, attribute)
    if not set_result then
        return "Error: Failed to set value for new key " .. new_key .. " - set_result is nil\n"
    end
    if set_result.err then
        return "Error: " .. set_result.err .. "\n"
    end
    redis.pcall('DEL', old_key) -- Delete old_key
    return "Success: updated " .. old_key .. "\n"
end

local function main()
    local result = "Success: return value "
    for channel = 1, number_of_channels do
        local channel_result = do_upgrade_channel(channel)
        if channel_result:sub(1, 6) == "Error:" then
            return channel_result -- Exit early if an error occurs
        end
        local interface_result = do_upgrade_interface(channel)
        if interface_result:sub(1, 6) == "Error:" then
            return interface_result -- Exit early if an error occurs
        end
        result = result .. channel_result .. "\n"
    end
    return result
end

return main()