#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -r, --runningversion    <release version>        runnning release version
  -i, --installingversion <release version>        installing release version
  -d, --directory         <path>                   path to persistent directory to update
  -h, --help                                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--runningversion")      set -- "$@" "-r" ;;
    "--installingversion")   set -- "$@" "-i" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

LOG_PREFIX="Handling mmc folder:"
RUNNING_RELEASE_VERSION=''
INSTALLING_RELEASE_VERSION=''
[ -z "${RESET_CONFIG}" ] && RESET_CONFIG='false'
REDIS_PORT=50001

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?r:i:d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    r)  RUNNING_RELEASE_VERSION=$OPTARG
      ;;
    i)  INSTALLING_RELEASE_VERSION=$OPTARG
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ -z $RUNNING_RELEASE_VERSION ]
then
  LOG "${LOG_PREFIX} failed to get the running release version"
  exit 1
fi

if [ -z $INSTALLING_RELEASE_VERSION ]
then
  LOG "Handling mmc folder: failed to get the installing release version"
  exit 1
fi

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "${LOG_PREFIX} persistent dir $PERSISTENT_DIRECTORY does not exist."
  exit 1
else
  LOG "${LOG_PREFIX} persistent dir : $PERSISTENT_DIRECTORY."
fi

function ping_redis() {
  timer=10
  redis_port=${1}
  is_up=false
  while true; do
      # Check if redis is up
      status=$(redis-cli -p ${redis_port} PING)
      if [[ "${status}" == 'PONG' ]]; then
          is_up=true
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for redis... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
        is_up=false
        break
      fi
      sleep 1
  done
  echo ${is_up}
}

function wait_redis_persistent() {
  timer=20
  while true; do
      # Check if the write operation is finished
      status=$(redis-cli -p ${REDIS_PORT} info persistence | grep -i aof_rewrite_in_progress)
      if [[ ${status} =~ ':0' ]]; then
          LOG "Redis aof_rewrite ready!"
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for aof_rewrite_in_progress... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
          LOG "aof_rewrite_in_progress timeout!"
          exit 1
      fi
      sleep 1
  done
}

# Handle symlink for redis database files
SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)
IMAGE_VERSION_FILE=${SCRIPTS_DIR}/../image/image_version.txt
IMAGE_VERSION_HASH=$(cat ${IMAGE_VERSION_FILE} | sha1sum | cut -c -40)
MMC_DEST_RELEASE="/mnt/mmc/sw_storage"
MMC_DEST_NEW="${MMC_DEST_RELEASE}/v${INSTALLING_RELEASE_VERSION}-${IMAGE_VERSION_HASH}/"
MMC_REDIS_NEW="${MMC_DEST_NEW}/redis/"
MMC_LINK_OLD="/var/persistent/etc/sw_storage"
MMC_LINK_NEW="${PERSISTENT_DIRECTORY}/etc/sw_storage"

# Cleanup old directories
MAX=30
COUNT=$(ls ${MMC_DEST_RELEASE} | wc -l)
if [[ ${COUNT} -gt ${MAX} ]]; then
  OBSOLETE_COUNT=$((COUNT - MAX))
  DIRS_NAME=$(ls -t ${MMC_DEST_RELEASE} | tail -n ${OBSOLETE_COUNT})
  for NAME in ${DIRS_NAME}; do
    FULL_PATH="${MMC_DEST_RELEASE}/${NAME}"
    LOG "${LOG_PREFIX} removing old dir: ${FULL_PATH}!"
    rm -rf ${FULL_PATH}
  done
fi

[[ -d ${MMC_DEST_NEW} ]] && rm -rf ${MMC_DEST_NEW}/*
mkdir -p ${MMC_REDIS_NEW}
if [[ $RESET_CONFIG == 'false' ]]; then
  if [[ -d ${MMC_LINK_OLD} ]] && [[ ! -f /usr/sbin/smartmanager/smartmanager.js ]]; then
    LOG "${LOG_PREFIX} old configurations will be kept!"
    PING_REDIS=`ping_redis ${REDIS_PORT}`
    if [[ ${PING_REDIS} == true ]]; then
      # Safe way to copy redis db file according to: https://redis.io/docs/manual/persistence/#backing-up-aof-persistence
      old_rewrite=$(redis-cli -p ${REDIS_PORT} CONFIG get auto-aof-rewrite-percentage | xargs | awk '{print $2}')
      redis-cli -p ${REDIS_PORT} CONFIG SET auto-aof-rewrite-percentage 0
      wait_redis_persistent
      cp -r ${MMC_LINK_OLD}/* ${MMC_DEST_NEW}
      redis-cli -p ${REDIS_PORT} CONFIG SET auto-aof-rewrite-percentage ${old_rewrite}
    else
      cp -r ${MMC_LINK_OLD}/* ${MMC_DEST_NEW}
    fi
  else
    LOG "${LOG_PREFIX} old configurations not fould!"
  fi
else
  LOG "${LOG_PREFIX} old configurations will be resetted!"
fi
cp -f ${IMAGE_VERSION_FILE} ${MMC_DEST_NEW}
rm -rf ${MMC_LINK_NEW}
mkdir -p ${PERSISTENT_DIRECTORY}/etc/
ln -fs ${MMC_DEST_NEW} ${MMC_LINK_NEW}


exit 0
